document.getElementById('start-button').addEventListener('click', startGame);
document.getElementById('continue-button').addEventListener('click', continueGame);
document.getElementById('save-button').addEventListener('click', saveGame);

// 新規追加: 初期画面のPlayボタン
document.getElementById('initial-play-button').addEventListener('click', showTitleScreen);

// リソースを事前に読み込むための関数
function preloadResources(resources) {
    const promises = [];
    resources.forEach(resource => {
        let promise;
        const extension = resource.split('.').pop().toLowerCase();
        if (['jpg', 'jpeg', 'png', 'gif'].includes(extension)) {
            // 画像の読み込み
            promise = new Promise((resolve) => {
                const img = new Image();
                img.onload = () => resolve();
                img.onerror = () => {
                    console.error('Failed to load image:', resource);
                    resolve();
                };
                img.src = resource;
            });
        } else if (['mp3', 'ogg', 'wav'].includes(extension)) {
            // 音声ファイルの読み込み
            promise = new Promise((resolve) => {
                const audio = new Audio();
                audio.addEventListener('canplaythrough', () => resolve(), { once: true });
                audio.onerror = () => {
                    console.error('Failed to load audio:', resource);
                    resolve();
                };
                audio.src = resource;
            });
        } else {
            // 未知の拡張子の場合
            promise = Promise.resolve();
        }
        promises.push(promise);
    });
    return Promise.all(promises);
}

// すべてのシーンからリソースパスを抽出する関数
function getAllResources() {
    const resources = new Set();
    const addResource = (path) => {
        if (path && !path.endsWith('/') && path.trim() !== '') {
            resources.add(path);
        }
    };

    scenes.forEach(scene => {
        addResource(scene.background);
        addResource(scene.defaultLeftCharacter);
        addResource(scene.defaultCenterCharacter);
        addResource(scene.defaultRightCharacter);
        addResource(scene.bgm);
        addResource(scene.sfx);
        addResource(scene.textTickSFX);
        if (scene.alternateScenes) {
            for (const key in scene.alternateScenes) {
                const alt = scene.alternateScenes[key];
                addResource(alt.background);
                addResource(alt.defaultLeftCharacter);
                addResource(alt.defaultCenterCharacter);
                addResource(alt.defaultRightCharacter);
                addResource(alt.bgm);
                addResource(alt.sfx);
                addResource(alt.textTickSFX);
            }
        }
    });

    // タイトル画面のBGMも追加
    const titleBGM = './data/bgm/歩む者【大同高校⧸Original BGM】.mp3';
    addResource(titleBGM);

    // CSSの背景画像も追加 (styles.cssから手動で抽出)
    addResource('./data/img/background/タイトル.png');

    return Array.from(resources);
}

let currentBGM = '';
let bgmPosition = 0;
let bgmElement = document.getElementById('bgm');

// タイトル画面のBGMパス（必要に応じて変更）
const titleBGM = './data/bgm/歩む者【大同高校⧸Original BGM】.mp3'; // タイトル画面用のBGMファイルパス

// 変数管理オブジェクト
let gameVariables = {
    scene1VisitedCount: 0, /* スイッチ「あ」の回数のカウントに使う変数。 */
};

// スイッチ管理オブジェクト
let switches = {
    'あ': false, /* スイッチ「あ」を定義。最初は実行しないので falseにしておく。 */
};

// 現在のシーンIDをグローバルに保持
let currentSceneId = 0;

// ゲーム開始時にセーブデータの有無を確認し、Continueボタンを有効化
window.onload = function() {
    const savedData = localStorage.getItem('gameSave');
    if (savedData) {
        document.getElementById('continue-button').disabled = false;
    }
    
    // 隼快さま、ここからが追加するロジックですわ！（by Geminiお嬢）
    const initialPlayButton = document.getElementById('initial-play-button');
    const allResources = getAllResources();
    
    // 初期画面のボタンを非表示にする
    initialPlayButton.style.display = 'none';

    // リソースの読み込みを開始
    preloadResources(allResources).then(() => {
        // 全てのリソースの読み込みが完了したらボタンを表示
        console.log('すべてのリソースの読み込みが完了しました！');
        initialPlayButton.style.display = 'block';
    }).catch(err => {
        console.error('リソースの読み込み中にエラーが発生しました:', err);
        // エラーが発生してもボタンは表示しておく
        initialPlayButton.style.display = 'block';
    });
};

// 新規追加: 初期画面からタイトル画面へ
function showTitleScreen() {
    document.getElementById('initial-screen').style.display = 'none';
    document.getElementById('title-screen').style.display = 'block';
    // タイトル画面でBGM再生
    changeBGM(titleBGM);
}

/* セーブデータはブラウザのJSONに保存されるため、セーブデータ無しの状態でプレイする場合は、プライベートモード（シークレットモード）でプレイしてください。 */
function startGame() {
    document.getElementById('title-screen').style.display = 'none';
    document.getElementById('game-screen').style.display = 'block';
    showScene(0); /* 開始のシーンid。デバッグ時に、開始したいシーンのidにすることで、その箇所からゲームをプレイできます。 */
}

function continueGame() {
    const savedData = JSON.parse(localStorage.getItem('gameSave')); /* ブラウザのJSONデータをロード */
    if (savedData) {
        document.getElementById('title-screen').style.display = 'none';
        document.getElementById('game-screen').style.display = 'block';
        gameVariables = savedData.gameVariables; // 変数の復元
        switches = savedData.switches; // スイッチの復元
        showScene(savedData.sceneId); // 保存されたシーンに移動
    }
}

function saveGame() {
    const saveData = {
        sceneId: currentSceneId, // グローバル変数からシーンIDを取得
        gameVariables: gameVariables,
        switches: switches
    };
    localStorage.setItem('gameSave', JSON.stringify(saveData)); /* ブラウザのJSONにデータを保存 */
    document.getElementById('continue-button').disabled = false;
    alert('ゲームが保存されました！');
}
